#include "View.h"

#include <wx/string.h>
#include <wx/frame.h>
#include <wx/msgdlg.h>
#include <wx/dcclient.h>
#include <wx/busyinfo.h>
#include <wx/fileconf.h>

#include <GL/glu.h>
#include <XFaceApp/Task.h>
using XFaceApp::Task;

//Event table for View
BEGIN_EVENT_TABLE(View, wxGLCanvas)
	EVT_SIZE(View::OnSize) 
    EVT_PAINT(View::OnPaint)
	EVT_ERASE_BACKGROUND(View::OnEraseBackground)
	EVT_IDLE(View::OnIdle)
END_EVENT_TABLE()

//Constructor for View class
View::View(wxWindow *parent, const Frame* frame, wxWindowID id, const wxPoint& pos, const wxSize& size, long style, const wxString& name)
:wxGLCanvas(parent, (wxGLCanvas*) NULL, id, pos, size, style, name ), _camera(0), _init(false), _face(0), _frame(frame)
{
	wxConfigBase *pConfig = wxConfigBase::Get();
    wxString dummy;
	pConfig->Read(_T("MPEG-4/Initial-FAP"), &dummy, wxT("Faps/joy.fap"));
	_initialFAP = dummy;
	pConfig->Read(_T("MPEG-4/Initial-FDP"), &dummy, wxT("alice.fdp"));
	_initialFDPFileName = dummy;
	pConfig->Read(_T("MPEG-4/Initial-FDP-Path"), &dummy, wxT("Faces/alice/"));
	_initialFDPPath = dummy;
	
	SetCurrent();

	InitGL();
	//_init = true;
	LoadFace();
}

View::~View(void)
{
	delete _camera;
	delete _face;
}

#pragma region GraphicEvents

//Called when the face is idle
void View::OnIdle(wxIdleEvent& event)
{
	if(!_face)
	{
		return;
	}
	
	SetCurrent();
	_face->processTask();
}

//Called when their is a size change event
void View::OnSize(wxSizeEvent& event)
{
    wxGLCanvas::OnSize(event);

	int width, height;
    GetClientSize(&width, &height);
#ifndef __WXMOTIF__
    if (GetContext())
#endif
    {
        SetCurrent();
 
		//call gl functions
		glViewport( 0, 0, width, height);
		glMatrixMode( GL_PROJECTION );
		glLoadIdentity( );
	    
		float ratio = (float)width / (float)height;
		gluPerspective( 30.0, ratio, 10.0, 1000.0 );
		glMatrixMode( GL_MODELVIEW );

		//if their is no camera object
		if (!_camera)
		{
			//create a new one
			_camera = new XFaceApp::ModelCamera;
			_camera->setScreenSize(width, height);
			_camera->setDistance(-700);
			_camera->setMode(XFaceApp::ModelCamera::ZOOM);
		}
		else
		{
			//adjust the current one
			_camera->setScreenSize(width, height);
			_camera->apply();
		}
    }
	Refresh();
}

//Called when their is a paint event
void View::OnPaint( wxPaintEvent& event )
{
    wxPaintDC dc(this);

#ifndef __WXMOTIF__
    if (!GetContext()) return;
#endif

    SetCurrent();

	//if the face hasn't been initialized
    if (!_init)
    {
		//initialize the face
        InitGL();
        _init = true;
		LoadFace();
    }
	else
	{
		//render the face
		Render();
	}
}

//Called when the face needs to be initialized
void View::InitGL()
{
	//call gl functions
    glShadeModel( GL_SMOOTH );
    glCullFace( GL_BACK );
    glFrontFace( GL_CCW );
    glEnable( GL_CULL_FACE );

	//read from the config file
	wxConfigBase *pConfig = wxConfigBase::Get();
    long red, green, blue;
	pConfig->Read(_T("Window/BackColor-R"), &red, 255);
	pConfig->Read(_T("Window/BackColor-G"), &green, 255);
	pConfig->Read(_T("Window/BackColor-B"), &blue, 255);
	
	glClearColor(red/1.0f, green/1.0f, blue/1.0f, 1.0f);
	glEnable(GL_DEPTH_TEST);
	glEnable(GL_LIGHTING);
	glEnable(GL_LIGHT0);
	glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
    glMatrixMode( GL_MODELVIEW );

	SwapBuffers();
}

//Called when the face needs to be rendered
void View::Render()
{
	if(!_face)
	{
		return;
	}

	//if their is no camera object
	if (!_camera)
	{
		//set-up camera object
		int width, height;
	    GetClientSize(&width, &height);

		glViewport( 0, 0, width, height);
		glMatrixMode( GL_PROJECTION );
		glLoadIdentity( );
	    
		float ratio = (float)width / (float)height;
		gluPerspective( 30.0, ratio, 10.0, 1000.0 );
		glMatrixMode( GL_MODELVIEW );

		//create new camera object
		_camera = new XFaceApp::ModelCamera;
		_camera->setScreenSize(width, height);
		_camera->setDistance(-700);
		_camera->setMode(XFaceApp::ModelCamera::ZOOM);
	}

	_camera->apply();

	Task renderTask("RENDER_FRAME");
	_face->newTask(renderTask);
}

#pragma endregion

#pragma region FaceEvents

//Called when the face needs to be loaded
void View::LoadFace()
{
	delete _face;
	_face = new XFaceApp::Face(this);

	//if the face doesn't initialize
	if (!_face->init())
	{
		delete _face;
		_face = 0;
		return;
	}

	//load initial paths for the face
	LoadFDP(_initialFDPFileName, _initialFDPPath);
	LoadFAP(_initialFAP);
}

//Called when a new FAP needs to be loaded
void View::LoadFAP(const std::string& filename)
{
	if(!_face)
	{
		return;
	}

	//load FAP by pushing path to file to the queue
	Task fapTask("LOAD_FAP");
	fapTask.pushParameter(filename);
	_face->newTask(fapTask);

	Refresh();
}

//Called when a new FDP needs to be loaded
void View::LoadFDP(const std::string& filename, const std::string& path)
{
	if(!_face)
	{
		return;
	}

	//pushes FDP path and filename to the queue
	Task fdpTask("LOAD_FDP");
	fdpTask.pushParameter(filename);
	fdpTask.pushParameter(path);
	_face->newTask(fdpTask);
	_face->processTask();

	//if there is a camera object
	if(_camera)
	{
		//adjust camera
		_camera->setAxisAngle(_face->getFace()->getFDP()->getGlobalAxisAngle());
		_camera->setTranslation(_face->getFace()->getFDP()->getGlobalTranslation());
	}

	Refresh();
}

//Called when the Talking Head needs to be stopped
void View::StopPlayback()
{
	if(!_face)
	{
		return;
	}

	Task renderTask("STOP_PLAYBACK");
	_face->newTask(renderTask);
}

//Called when the Talking Head needs to be played
void View::StartPlayback()
{
	if(!_face)
	{
		return;
	}

	Task playTask("RESUME_PLAYBACK");
	_face->newTask(playTask);
}

#pragma endregion

//Called when the background color needs to be set
void View::SetBackgroundColour(wxColour col) 
{
	SetCurrent();
	glClearColor((float)col.Red() / 255.0f, (float)col.Green() / 255.0f, (float)col.Blue() / 255.0f, 1.0f);
	Refresh();
}
